﻿using System.ComponentModel;
using System.Windows.Input;
//using AVFoundation;
using Microsoft.Maui.Dispatching;
using Plugin.Maui.Audio;

namespace DirtbagSampler.ViewModels;

public class MusicPlayerPageViewModel : BaseViewModel, IQueryAttributable, IDisposable
{
	readonly IAudioManager audioManager;
	readonly IDispatcher dispatcher;
	IAudioPlayer audioPlayer0;
	IAudioPlayer audioPlayer1;
	IAudioPlayer audioPlayer2;
	IAudioPlayer audioPlayer3;
	IAudioPlayer[] audioPlayers;
	//AudioPlayer audioPlayer4;
	TimeSpan animationProgress;
	MusicItemViewModel musicItemViewModel;
	bool isPositionChangeSystemDriven;
	bool isDisposed;

	public MusicPlayerPageViewModel(
		IAudioManager audioManager,
		IDispatcher dispatcher)
	{
		this.audioManager = audioManager;
		this.dispatcher = dispatcher;



		PlayCommand0 = new Command(() => Play(0));
		PlayCommand1 = new Command(() => Play(1));
		PlayCommand2 = new Command(() => Play(2));
		PlayCommand3 = new Command(() => Play(3));
		PauseCommand = new Command(Pause);
		StopCommand = new Command(Stop);

		PlayCommand = new Command<int>(i => Play(i));

	}




	public async void ApplyQueryAttributes(IDictionary<string, object> query)
	{
		if (query.TryGetValue(Routes.MusicPlayer.Arguments.Music, out object musicObject) &&
			musicObject is MusicItemViewModel musicItem)
		{
			MusicItemViewModel = musicItem;

			// Create a list of all files you want to play
			List<string> mp3_fn_list = new List<string> { musicItem.Filename0, musicItem.Filename1, musicItem.Filename2, musicItem.Filename3 };

			IAudioPlayer[] audioPlayers = new IAudioPlayer[(int)mp3_fn_list.Count];

			for (int i = 0; i < (int)mp3_fn_list.Count; i++)
			{
				audioPlayers[i] = audioManager.CreatePlayer(
					await FileSystem.OpenAppPackageFileAsync(mp3_fn_list[i])
					);
				audioPlayers[i].Loop = false;
			}

			audioPlayer0 = audioManager.CreatePlayer(
				await FileSystem.OpenAppPackageFileAsync(mp3_fn_list[0]));
			audioPlayer1 = audioManager.CreatePlayer(
				await FileSystem.OpenAppPackageFileAsync(mp3_fn_list[1]));
			audioPlayer2 = audioManager.CreatePlayer(
				await FileSystem.OpenAppPackageFileAsync(mp3_fn_list[2]));
			audioPlayer3 = audioManager.CreatePlayer(
				await FileSystem.OpenAppPackageFileAsync(mp3_fn_list[3]));

			NotifyPropertyChanged(nameof(HasAudioSource));
			NotifyPropertyChanged(nameof(Duration));
			NotifyPropertyChanged(nameof(CanSetSpeed));
			NotifyPropertyChanged(nameof(MinimumSpeed));
			NotifyPropertyChanged(nameof(MaximumSpeed));

		}
	}

	public double CurrentPosition
	{
		get => audioPlayer0?.CurrentPosition ?? 0;
		set
		{
			if (audioPlayer0 is not null &&
				audioPlayer0.CanSeek &&
				isPositionChangeSystemDriven is false)
			{
				audioPlayer0.Seek(value);
			}
		}
	}


	public double Duration => audioPlayer0?.Duration ?? 1;

	public MusicItemViewModel MusicItemViewModel
	{
		get => musicItemViewModel;
		set
		{
			musicItemViewModel = value;
			NotifyPropertyChanged();
		}
	}

	public bool HasAudioSource => audioPlayer0 is not null;

	public bool IsPlaying => audioPlayer0?.IsPlaying ?? false;

	public TimeSpan AnimationProgress
	{
		get => animationProgress;
		set
		{
			animationProgress = value;
			NotifyPropertyChanged();
		}
	}

	public Command PlayCommand0 { get; set; }
	public Command PlayCommand1 { get; set; }
	public Command PlayCommand2 { get; set; }
	public Command PlayCommand3 { get; set; }

	public Command PauseCommand { get; set; }
	public Command StopCommand { get; set; }

	public ICommand PlayCommand { get; set; }

	public class Player
	{
		public Command PlayFile { get; set; }
		public string filename { get; set; }

		public void Play()
		{
			// Your code goes here to execute the PlayFile command, passing in the filename as a parameter
		}
	}

	public double Volume
	{
		get => audioPlayer0?.Volume ?? 1;
		set
		{
			if (audioPlayer0 != null)
			{
				audioPlayer0.Volume = value;
			}
			if (audioPlayer1 != null)
			{
				audioPlayer1.Volume = value;
			}
			if (audioPlayer2 != null)
			{
				audioPlayer2.Volume = value;
			}
			if (audioPlayer3 != null)
			{
				audioPlayer3.Volume = value;
			}
		}
	}

	public bool CanSetSpeed => audioPlayer0?.CanSetSpeed ?? false;

	public double Speed
	{
		get => audioPlayer0?.Speed ?? 1;
		set
		{
			try
			{
				if (audioPlayer0?.CanSetSpeed ?? false)
				{
					audioPlayer0.Speed = Math.Round(value, 1, MidpointRounding.AwayFromZero);
					NotifyPropertyChanged();
				}
			}
			catch (Exception ex)
			{
				App.Current.MainPage.DisplayAlert("Speed", ex.Message, "OK");
			}
		}
	}

	public double MinimumSpeed => audioPlayer0?.MinimumSpeed ?? 1;
	public double MaximumSpeed => audioPlayer0?.MaximumSpeed ?? 1;

	public bool Loop
	{
		get => audioPlayer0?.Loop ?? false;
		set
		{
			audioPlayer0.Loop = value;
		}
	}

	public async void Play0(string soundFilename)
	{
		Stop();
		audioPlayer0 = audioManager.CreatePlayer(
			await FileSystem.OpenAppPackageFileAsync(soundFilename));
		audioPlayer0.Play();

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}
	public void Play2(IAudioPlayer i)
	{
		//Stop2();
//		audioPlayers[i] = audioManager.CreatePlayer(
//			await FileSystem.OpenAppPackageFileAsync(soundFilename));
		i.Play();

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}

	void Play1()
	{
		Stop();

		audioPlayer1.Play();

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}

	void Play21()
	{
		Stop();
		audioPlayer2.Play();

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}
	void Play3()
	{
		Stop();
		audioPlayer3.Play();

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}

	void Play(int playerNumber)
	{
		Stop();
		switch (playerNumber)
		{
			case 0:
				audioPlayer0.Play();
				break;
			case 1:
				audioPlayer1.Play();
				break;
			case 2:
				audioPlayer2.Play();
				break;
			case 3:
				audioPlayer3.Play();
				break;
		}

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}


	void Pause()
	{
		if (audioPlayer0.IsPlaying)
		{
			audioPlayer0.Pause();
		}
		else
		{
			audioPlayer0.Play();
		}

		UpdatePlaybackPosition();
		NotifyPropertyChanged(nameof(IsPlaying));
	}

	void Stop()
	{
		if ((audioPlayer0 is IAudioPlayer) && (audioPlayer0.IsPlaying))
		{
			audioPlayer0.Stop();

			AnimationProgress = TimeSpan.Zero;

			NotifyPropertyChanged(nameof(IsPlaying));
		}
		if ((audioPlayer1 is IAudioPlayer) && (audioPlayer1.IsPlaying))
		{
			audioPlayer1.Stop();

			AnimationProgress = TimeSpan.Zero;

			NotifyPropertyChanged(nameof(IsPlaying));
		}
		if ((audioPlayer2 is IAudioPlayer) && (audioPlayer2.IsPlaying))
		{
			audioPlayer2.Stop();

			AnimationProgress = TimeSpan.Zero;

			NotifyPropertyChanged(nameof(IsPlaying));
		}
		if ((audioPlayer3 is IAudioPlayer) && (audioPlayer3.IsPlaying))
		{
			audioPlayer3.Stop();

			AnimationProgress = TimeSpan.Zero;

			NotifyPropertyChanged(nameof(IsPlaying));
		}
	}

	void Stop2()
	{
		foreach (IAudioPlayer i in audioPlayers)
		{
			if ((i is IAudioPlayer) && (i.IsPlaying))
			{
				i.Stop();

				AnimationProgress = TimeSpan.Zero;

				NotifyPropertyChanged(nameof(IsPlaying));
			}
		}
	}

	void UpdatePlaybackPosition()
	{
		if (audioPlayer0?.IsPlaying is false)
		{
			return;
		}

		dispatcher.DispatchDelayed(
			TimeSpan.FromMilliseconds(16),
			() =>
			{
				Console.WriteLine($"{CurrentPosition} with duration of {Duration}");

				isPositionChangeSystemDriven = true;

				NotifyPropertyChanged(nameof(CurrentPosition));

				isPositionChangeSystemDriven = false;

				UpdatePlaybackPosition();
			});
	}

	public void TidyUp()
	{
		audioPlayer0?.Dispose();
		audioPlayer0 = null;
		audioPlayer1?.Dispose();
		audioPlayer1 = null;
		audioPlayer2?.Dispose();
		audioPlayer2 = null;
		audioPlayer3?.Dispose();
		audioPlayer3 = null;

		if (audioPlayers != null) {
			for (int i = 0; i < (int)audioPlayers.Length; i++)
			{
				audioPlayers[i].Dispose();
				audioPlayers[i] = null;
			}
		}
		audioPlayers = null;
	}

	~MusicPlayerPageViewModel()
	{
		Dispose(false);
	}

	public void Dispose()
	{
		Dispose(true);

		GC.SuppressFinalize(this);
	}

	protected virtual void Dispose(bool disposing)
	{
		if (isDisposed)
		{
			return;
		}

		if (disposing)
		{
			TidyUp();
		}

		isDisposed = true;
	}
}
